/*
*  BayesTriats 4.0
*
*  copyright 2022
*
*  Andrew Meade
*  School of Biological Sciences
*  University of Reading
*  Reading
*  Berkshire
*  RG6 6BX
*
* BayesTriats is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>
*
*/



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>


#include "TypeDef.h"
#include "Trees.h"
#include "Rates.h"
#include "Priors.h"
#include "Likelihood.h"
#include "GenLib.h"
#include "RandLib.h"
#include "Options.h"
#include "RevJump.h"
#include "Data.h"
#include "Gamma.h"
#include "ML.h"
#include "VarRates.h"
#include "Threaded.h"
#include "Schedule.h"
#include "ModelFile.h"
#include "Stones.h"
#include "RJDummy.h"
#include "Contrasts.h"
#include "FatTail.h"
#include "Geo.h"
#include "TransformTree.h"
#include "Prob.h"
#include "MCMC.h"
#include "VarRatesMLLocalTransfom.h"


#include <gsl/gsl_rng.h>


#ifdef	 JNIRUN
//	extern void	SetProgress(JNIEnv *Env, jobject Obj, int Progress);
	#include "BayesTraitsJNI.h"
#endif

void	PrintPriorHeadder(FILE* Str, OPTIONS *Opt, RATES* Rates)
{
	int		PIndex;
	PRIOR	*Prior;

	for(PIndex=0;PIndex<Rates->NoPriors;PIndex++)
	{
		Prior = Rates->Priors[PIndex];

		if(Prior->UseHP == TRUE)
		{
			switch(Prior->Dist)
			{
				case PDIST_EXP:
					fprintf(Str, "%s - Mean\t", Prior->Name);
				break;

				case PDIST_GAMMA:
					fprintf(Str, "%s - Shape\t%s - Scale\t", Prior->Name, Prior->Name);
				break;

				case PDIST_LOGNORMAL:
					fprintf(Str, "%s - Mue\t%s - Sig\t", Prior->Name, Prior->Name);
				break;

				default:
					printf("%s - %s::%d Hyper Prior not supported.", Prior->Name, __FILE__, __LINE__);
					exit(0);
			}
		}
	}

	fprintf(Str, "\n");
}


void	UpDateHMean(OPTIONS *Opt, RATES *Rates)
{
#ifndef BIG_LH
	Rates->HMeanCount++;
	Rates->HMeanSum += 1/exp(Rates->Lh);
#else
	mpfr_t	t1, t2;

	Rates->HMeanCount++;

	mpfr_init2(t1, Opt->Precision);
	mpfr_init2(t2, Opt->Precision);

	mpfr_set_d(t1, Rates->Lh, DEF_ROUND);

	mpfr_exp(t2, t1, DEF_ROUND);

	mpfr_d_div(t1, 1.0, t2, DEF_ROUND);

	mpfr_add(t2, t1, Rates->HMeanSum, DEF_ROUND);

	mpfr_set(Rates->HMeanSum, t2, DEF_ROUND);

	mpfr_clears(t1, t2, NULL);
#endif
}

void	PrintPrior(FILE* Str, PRIOR *Prior)
{
	int	Index;

	for(Index=0;Index<DISTPRAMS[Prior->Dist];Index++)
		fprintf(Str, "%f\t", Prior->DistVals[Index]);
}

void	PrintMCMCSample(long long Itters, SCHEDULE* Shed, OPTIONS *Opt, RATES *Rates, FILE* Str)
{
	TREES*	Trees;
	int		PIndex;
	double	HMean;
	PRIOR	*Prior;

	Trees = Opt->Trees;

	HMean = GetHMean(Opt, Rates);
	fprintf(Str, "%lld\t%f\t%d\t", Itters, Rates->Lh, Rates->TreeNo+1);

	PrintRates(Str, Rates, Opt, Shed);

	for(PIndex=0;PIndex<Rates->NoPriors;PIndex++)
	{
		Prior = Rates->Priors[PIndex];
		if(Prior->UseHP == TRUE)
			PrintPrior(Str, Rates->Priors[PIndex]);
	}

	fprintf(Str, "\n");

	fflush(stdout);
}

void	PrintTest(int Itters, RATES* Rates)
{
	char	MType;
	int		Index;

	MType = RJModelType(Rates->MappingVect);

	printf("%d\t%d\t'", Itters, NoOfPramGroups(Rates, NULL, NULL));

	for(Index=0;Index<Rates->NoOfFullRates;Index++)
	{
		if(Rates->MappingVect[Index] == ZERO_RATE_NO)
			printf("Z");
		else
		{
			if(Rates->MappingVect[Index] <= 9)
				printf("%d", Rates->MappingVect[Index]);
			else
				printf("%c", Rates->MappingVect[Index] + 'A');
		}
	}
	printf("\n");
}

void	TestInitPho(OPTIONS *Opt, TREES *Trees, RATES *Rates)
{
	int Index, RIndex;

	for(Index=0;Index<1000;Index++)
	{
		for(RIndex=0;RIndex<Rates->NoOfRates;RIndex++)
		{
			Rates->Rates[RIndex] = RandDouble(Rates->RS) * 0.01;
		}

		Rates->Lh = Likelihood(Rates, Trees, Opt);

		printf("Lh\t%d\t%f\n", Index, Rates->Lh);

		fflush(stdout);
	}

	exit(0);
}

double	GetRandValFromType(TRANSFORM_TYPE Type, RATES *Rates, gsl_rng *RNG)
{
	PRIOR *Prior;

	Prior = NULL;

	if(Type == VR_KAPPA)
		Prior = GetPriorFromName("Kappa", Rates->Priors, Rates->NoPriors);

	if(Type == VR_LAMBDA)
		Prior = GetPriorFromName("Lambda", Rates->Priors, Rates->NoPriors);

	if(Type == VR_DELTA)
		Prior = GetPriorFromName("Delta", Rates->Priors, Rates->NoPriors);

	if(Type == VR_OU)
		Prior = GetPriorFromName("OU", Rates->Priors, Rates->NoPriors);

	if(Type == VR_NODE)
		Prior = GetPriorFromName("VRNode", Rates->Priors, Rates->NoPriors);

	if(Type == VR_BL)
		Prior = GetPriorFromName("VRBL", Rates->Priors, Rates->NoPriors);

	if(Type == VR_LS_BL)
		Prior = GetPriorFromName("VR_LS_BL", Rates->Priors, Rates->NoPriors);

	assert(Prior != NULL);

	return RandFromPrior(RNG, Prior);
}


void	SetDefMCMCParameters(OPTIONS *Opt, TREES *Trees, RATES *Rates, 	gsl_rng *RNG)
{
	int Index;
	LOCAL_TRANSFORM *LR;
	PRIOR *Prior;
	double PriorVal;


	if(Opt->EstKappa == TRUE)
		Rates->Kappa = GetRandValFromType(VR_KAPPA, Rates, RNG);

	if(Opt->EstLambda == TRUE)
		Rates->Lambda = GetRandValFromType(VR_LAMBDA, Rates, RNG);

	if(Opt->EstDelta == TRUE)
		Rates->Delta = GetRandValFromType(VR_DELTA, Rates, RNG);

	if(Opt->EstOU == TRUE)
		Rates->OU = GetRandValFromType(VR_OU, Rates, RNG);

	if(Opt->UseGlobalTrend == TRUE)
	{
		Prior = GetPriorFromName("GlobalTrend", Rates->Priors, Rates->NoPriors);
		Rates->GlobalTrend = RandFromPrior(RNG, Prior);
//		Rates->GlobalTrend = 0.0;
	}

	if(Opt->EstGamma == TRUE)
	{
		Prior = GetPriorFromName("Gamma", Rates->Priors, Rates->NoPriors);
		Rates->Gamma = RandFromPrior(RNG, Prior);
	}

	for(Index=0;Index<Rates->NoLocalTransforms;Index++)
	{
		LR = Rates->LocalTransforms[Index];
		if(LR->Est == TRUE)
			LR->Scale = GetRandValFromType(LR->Type, Rates, RNG);
	}

	if(Rates->NoEstData > 0 && Opt->DataType == CONTINUOUS)
		SetEstDataFromPrior(Rates);

	if(Opt->UseCovarion == TRUE)
	{
		Prior = GetPriorFromName("CVSwichRate",Rates->Priors,Rates->NoPriors);
		PriorVal = RandFromPrior(RNG, Prior);
		Rates->OffToOn = Rates->OnToOff = PriorVal;
	}
}

double		ValidMCMCParameters(OPTIONS *Opt, TREES *Trees, RATES *Rates)
{

	Rates->Lh = Likelihood(Rates, Trees, Opt);

	if(Rates->Lh == ERRLH)
		return ERRLH;

	CalcPriors(Rates, Opt);

	if(Rates->LhPrior == ERRLH)
		return ERRLH;

	return Rates->Lh + Rates->LhPrior;
}

void	SetAllMCMCRates(double Val, RATES *Rates)
{
	int Index;

	for(Index=0;Index<Rates->NoOfRates;Index++)
		Rates->Rates[Index] = Val;
}

int	FindValidStartRateAllSame(OPTIONS *Opt, TREES *Trees, RATES *Rates)
{
	double CRate, BRate;
	double CLh, BLh;

	SetAllMCMCRates(1.0, Rates);
	CLh = ValidMCMCParameters(Opt, Trees, Rates);


	CRate = RATE_MAX;
	BRate = -1.0;
	BLh = ERRLH;
	do
	{
		SetAllMCMCRates(CRate, Rates);
		CLh = ValidMCMCParameters(Opt, Trees, Rates);
		if(CLh != ERRLH && CLh > BLh)
		{
			BRate = CRate;
			BLh = CLh;
		}

		CRate = CRate * 0.1;

	} while(CRate > RATE_MIN);

	if(BLh == ERRLH)
		return FALSE;

	SetAllMCMCRates(BRate, Rates);
	Rates->Lh = ValidMCMCParameters(Opt, Trees, Rates);

	return TRUE;
}


double	RandFromPriorPosition(int Pos, OPTIONS *Opt, TREES *Trees, RATES *Rates, gsl_rng *RNG)
{
	PRIOR *P;

	if(Opt->UseRJMCMC == TRUE)
		P = GetPriorFromName("RJRates", Rates->Priors, Rates->NoPriors);
	else
		P =  GetPriorFromName(Rates->RateNames[Pos], Rates->Priors, Rates->NoPriors);;

	return RandFromPrior(RNG, P);
}

void	RandRatesFromPrior(OPTIONS *Opt, TREES *Trees, RATES *Rates, gsl_rng *RNG)
{
	int TNo, RIndex;

	for(TNo=0;TNo<NO_RAND_START_TRIES;TNo++)
	{
		if(gsl_rng_uniform(RNG) < 0.1)
			FindValidStartRateAllSame(Opt, Trees, Rates);
		{
			for(RIndex=0;RIndex<Rates->NoOfRates;RIndex++)
				Rates->Rates[RIndex] = RandFromPriorPosition(RIndex, Opt, Trees, Rates, RNG);
		}

		SetDefMCMCParameters(Opt, Trees, Rates, RNG);

		if(ValidMCMCParameters(Opt, Trees, Rates) != ERRLH)
			return;
	}

	printf("Cannot find initial starting set of parameters.");
	exit(0);
}

void FindValidStartLh(OPTIONS *Opt, TREES *Trees, RATES *Rates, gsl_rng *RNG)
{
	RandRatesFromPrior(Opt, Trees, Rates, RNG);
}

void	InitMCMC(OPTIONS *Opt, TREES *Trees, RATES *Rates)
{
	Rates->TreeNo = 0;

	SetDefMCMCParameters(Opt, Trees, Rates, Rates->RNG);

	if(Opt->MCMCMLStart == TRUE)
	{
		printf("Starting MCMC form ML is not avalable, please e-mail support.\n");
		exit(0);
		MLTree(Opt, Trees, Rates);
	}

	FindValidStartLh(Opt, Trees, Rates, Rates->RNG);

	if(Opt->VarRatesCheckPoint != NULL)
		SetVarRatesFromStr(Rates, Opt, Opt->VarRatesCheckPoint);
}

void	ShowTimeSec(double StartT, double EndT)
{
	printf("Sec:\t%f\n", EndT - StartT);
}


FILE*	SetScheduleFile(OPTIONS *Opt, SCHEDULE*	Shed)
{
	FILE *Ret;

	Ret = OpenWriteWithExt(Opt->BaseOutputFN, OUTPUT_EXT_SCHEDULE);

	PrintShedHeadder(Opt, Shed, Ret);


	return Ret;
}

FILE*	CreatStoneOuput(OPTIONS *Opt)
{
	FILE*	Ret;

	Ret = OpenWriteWithExt(Opt->BaseOutputFN, OUTPUT_EXT_STONES);

	OutputStoneHeadder(Ret, Opt->Stones);

	return Ret;
}

int		ExitMCMC(OPTIONS *Opt, long long Itters)
{
	if(Opt->Stones != NULL)
		return StoneExit(Opt->Stones, Itters);

	if((Opt->Itters == Itters) && (Opt->Itters != -1))
		return TRUE;

	return FALSE;
}


void	TestArea(OPTIONS *Opt, TREES *Trees, RATES *Rates)
{
//	int Index;
	double Lh, X;

	TestDummyCodeSig(Opt, Trees, Rates);
	exit(0);

	Lh = Likelihood(Rates, Trees, Opt);
	printf("Lh:\t%f\n", Lh);

	for(X=-1;X<2;X += 0.0001)
	{
		Rates->Rates[0] = X;
		Lh = Likelihood(Rates, Trees, Opt);
		printf("%f\t%f\n", X, Lh);
	}

	exit(0);
}

void	ReSetAccFlags(RATES *Rates)
{
	Rates->AutoAccept = FALSE;
	Rates->CalcLh = TRUE;
}

int		MCMCAccept(long long Itters, OPTIONS *Opt, TREES *Trees, SCHEDULE* Shed, RATES *CRates, RATES *NRates)
{
	double Heat;
		
//	printf("ItterOP:\t%llu\t%s\t%f\t%f\t%f\n", Itters, SHEDOP[Shed->Op], CRates->Lh - NRates->Lh, CRates->Lh, NRates->Lh);

	if(NRates->AutoAccept == TRUE)
	{
		ReSetAccFlags(NRates);
		return TRUE;
	}

	ReSetAccFlags(NRates);

	if((Shed->Op == S_FAT_TAIL_ANS_ALL || Shed->Op == S_FAT_TAIL_ANS) && Opt->Model == M_FATTAIL)
		return TRUE;

	Heat = NRates->Lh - CRates->Lh;

	if(Opt->Stones != NULL)
		Heat = GetStoneHeat(Opt->Stones, Itters, Heat);

	Heat += NRates->LhPrior - CRates->LhPrior;
	Heat += NRates->LnHastings;

	if(log(gsl_rng_uniform_pos(CRates->RNG)) <= Heat)
		return TRUE;

	return FALSE;
}

void PrintStdOutHeader(OPTIONS *Opt)
{
	PrintOptions(stdout, Opt);
	printf("Iteration\tLh\tLh Prior\tElapsed Seconds\tState");
	printf("\n");
	fflush(stdout);
}

void PrintChainState(OPTIONS *Opt,long long Itters, int BurnIn)
{
	if(BurnIn == FALSE)
	{
		printf("Burn In");
		return;
	}

	if(StonesStarted(Opt->Stones, Itters) == TRUE)
	{
		printf("Stepping Stone Sampler");
		return;
	}

	printf("Sampling");				
}


void PrintStdOut(OPTIONS *Opt, RATES *Rates, long long Itters, double Seconds, int BurnIn)
{
	printf("%zd\t%f\t%f\t%f\t", Itters, Rates->Lh, Rates->LhPrior, Seconds);

	PrintChainState(Opt, Itters, BurnIn);

	printf("\n");
	fflush(stdout);
}

int GetNodeDepth(NODE N)
{
	int Ret;

	Ret = 0;
	while(N->Ans != NULL)
	{
		N = N->Ans;
		Ret++;
	}

	return Ret;
}

void 	MCMCTest(OPTIONS *Opt, TREES *Trees, RATES*	Rates, SCHEDULE* Shed)
{
	double		StartT, EndT, Lh;
	TREE *Tree;
	int Index;

	Tree = Trees->Tree[0];

	Lh = Likelihood(Rates, Trees, Opt);

	printf("%f\n", Lh);

	StartT = GetSeconds();

	for(Index=0;Index<2500;Index++)
	{
		GeoUpDateAllAnsStates(Opt, Trees, Rates);
		Lh = Likelihood(Rates, Trees, Opt);
	}

	EndT = GetSeconds();

	Lh = Likelihood(Rates, Trees, Opt);
	
	printf("%d\t%f\n", Opt->Cores, Lh);

	printf("Total Time:\t%f\n", EndT - StartT);

	exit(0);


	for(Index=0;Index<Tree->NoParallelGroups;Index++)
	{
		printf("%d\t%d\n", Index, Tree->ParallelGroupSize[Index]);
	}
	exit(0);



	for(Index=0;Index<1000;Index++)
	{
	//	Likelihood(Rates, Trees, Opt);
		GeoUpDateAllAnsStates(Opt, Trees, Rates);
	}

	printf("Total Time:\t%f\n", EndT - StartT);

	exit(0);
}

#ifdef	 JNIRUN
	void	MCMC(OPTIONS *Opt, TREES *Trees, JNIEnv *Env, jobject Obj)
#else
	void	MCMC(OPTIONS *Opt, TREES *Trees)	
#endif
{
	RATES*		CRates;
	RATES*		NRates;
	long long	Itters;
	double		StartT, EndT;
	SCHEDULE*	Shed;
	FILE*		ShedFile;
	FILE*		SaveModelF;
	FILE*		StoneF;
	int			BurntIn, GBurntIn;

#ifdef	JNIRUN
	long		FP;
#endif

	ShedFile	= NULL;
	SaveModelF	= NULL;

	CRates	=	CreatRates(Opt);
	NRates	=	CreatRates(Opt);

	Shed = CreatSchedule(Opt, CRates->RS);
//	SetShedOpFreq(Shed, S_VARRATES_ADD_REMOVE, 0);

	if(Opt->ModelType == MT_FATTAIL)
	{
//		InitFatTailRates(Opt, Trees, CRates);
		InitFattailFile(Opt, Trees);
	}

	if(UseNonParametricMethods(Opt) == TRUE)
		InitVarRatesFiles(Opt, Trees, CRates);

	if(Opt->RJDummy == TRUE)
		InitRJDummyFile(Opt);

	#ifndef JNIRUN
		PrintStdOutHeader(Opt);
//		PrintRatesHeadder(stdout, Opt);
//		PrintPriorHeadder(stdout, Opt, CRates);
	#endif

	PrintOptions(Opt->LogFile, Opt);

	#ifdef JNIRUN
		fflush(Opt->LogFile);
		FP = ftell(Opt->LogFile);
/*		GotoFileEnd(Opt->LogFileRead, Opt->LogFileBuffer, LOGFILEBUFFERSIZE); */
	#endif

	PrintRatesHeadder(Opt->LogFile, Opt);
	PrintPriorHeadder(Opt->LogFile, Opt, CRates);
	fflush(Opt->LogFile);

	#ifdef JNIRUN
		fflush(Opt->LogFile);
		fseek(Opt->LogFileRead, FP, SEEK_SET);
		fgets(Opt->LogFileBuffer, LOGFILEBUFFERSIZE, Opt->LogFileRead);
		ProcessHeaders(Env, Obj, Opt);
	#endif

	InitMCMC(Opt, Trees, CRates);

//	VarRatesMLLocalTransfom(Opt, Trees, CRates);

//	LoadGeoData(Opt, Trees, CRates, "64500000	-25475.320861	2.000000	476894.245712	-12.241430	-54.372313	-29.374763	-53.392316	-31.334839	-10.315918	-23.530626	-61.735250	-28.114094	-53.694947	-23.820705	-54.168004	1.021989	-52.100548	-2.206263	12.535661	-1.608834	14.459407	-4.302661	12.545568	-4.420746	-48.072172	-31.724486	-47.024754	-0.891350	-44.566302	-54.293652	50.872424	-1.325818	-44.483307	-1.419007	-44.473742	-1.297270	-44.416480	-6.737613	-28.580830	-9.523549	-18.427069	7.672958	16.344592	9.863560	25.900947	7.867131	32.911087	-49.870367	53.433011	-21.674372	45.463312	117.931232	19.799804	11.464746	43.389569	5.988624	37.199876	15.336518	52.190656	-47.915134	53.296391	-60.684210	42.349857	83.204407	46.318610	98.360462	41.263718	98.817109	41.354998	101.575021	40.005429	109.921719	44.226090	108.741927	44.518287	112.080048	46.207823	112.072375	46.221157	114.570236	49.024798	109.198514	43.229984	-79.235454	56.788108	107.340814	43.545189	105.346246	42.504104	104.027005	42.338974	104.727786	42.758098	94.801157	44.492894	-53.241945	34.715980	-53.501609	35.290647	-54.719204	36.301472	11.285120	42.686324	-51.110808	37.233261	-53.983467	37.611575	-53.979383	38.005905	-58.507055	41.941458	-59.926560	41.488940	-64.559652	38.071482	126.487389	43.109325	-62.158308	37.018600	-63.059413	39.934744	-62.916286	40.106081	-78.270786	55.218528	-79.467812	55.309942	-4.355717	-22.328095	125.147170	44.356423	115.983483	39.526680	111.233224	37.767544	117.825627	41.688626	99.177716	39.893173	96.846830	42.405929	91.355656	46.384545	91.371140	44.028781	-76.034987	55.559357	116.282032	41.799918	115.362331	42.278848	115.628402	41.106625	106.225696	43.255798	107.786229	43.031594	104.715454	37.982993	102.754258	37.532511	102.417766	38.410511	101.382468	38.014201	124.455955	41.926551	122.761867	45.698680	-74.025639	68.820582	-78.040869	59.005257	-75.695214	58.422174	-76.009595	57.792190	-77.933444	56.469321	-72.184406	54.710149	-72.304535	54.529816	106.631373	47.746217	109.443041	56.350721	89.988622	44.961861	-79.813561	47.874283	-81.131554	51.902196	-77.101694	56.803861	-77.508471	56.727208	-76.784674	56.918542	-76.633419	57.741747	-77.442358	56.998816	-77.791264	57.146058	-78.754358	56.779501	-78.366194	58.392937	-77.303262	43.537763	-75.323990	57.301913	-74.006076	43.044292	-77.843347	48.489784	-79.570515	46.004061	-73.073633	44.770631	-76.224000	44.835011	-76.742855	48.539620	-80.970093	47.078827	-78.693868	49.268991	-84.015783	51.229034	-78.890305	48.063134	-79.963810	51.738603	-36.769053	50.494055	-51.568818	44.051820	-47.614942	44.528276	-67.171729	47.583657	-70.181403	45.740287	-34.084986	45.226924	-35.362445	43.796333	-60.357672	56.400175	115.800556	52.321031	126.273746	48.280901	-37.280848	50.921778	-35.319233	44.622354	-30.182431	41.410338	-49.136702	-44.538674	-31.975548	39.608901	17.914937	27.490817	20.999243	30.971254	18.330421	35.067413	20.442339	31.692450	-35.124875	38.116802	-52.173945	32.853610	-32.674496	36.064607	-20.701976	37.558906	-50.347474	36.090128	-53.402178	36.055808	-53.306514	34.855839	-59.804106	38.168699	-59.829250	38.419042	-56.588582	41.333520	-47.801353	44.502021	-50.659036	43.107019	8.191991	47.064086	14.032516	41.970274	19.124608	34.619535	105.589790	48.167338	112.218461	46.629641	104.861952	47.309797	108.530282	45.166519	104.800865	43.195163	107.235534	37.811602	110.519776	39.380642	114.350327	41.989118	114.193012	37.826993	122.109118	38.910279	116.384332	40.931882	118.203042	39.527329	119.897676	44.083767	-57.426244	47.803410	-57.765559	45.128504	-63.229957	49.931486	-61.286702	53.378746	-76.349080	45.084536	-81.424488	51.101030	-78.949211	51.912147	-77.048935	52.886454	-81.220909	49.436458	-77.950059	54.833814	-79.285414	54.734795	-83.008427	58.098587	-81.946175	55.939674	-78.969822	55.385550	-49.612422	-39.872386	-77.300310	56.363084	-89.001582	57.905822	-84.308980	61.763195	-87.976429	57.827100	119.900799	52.396508	-84.535714	61.266711	-92.688215	55.378788	-56.363768	53.113295	-62.267757	50.012233	104.498843	30.262673	-69.483208	49.150982	-64.436785	52.793069	-0.658522	33.921926	-66.619832	53.202263	-81.293864	50.810347	-81.438368	49.401424	-73.116979	52.934071	116.811287	50.308601	-75.293460	57.466395	-74.939518	58.138787	-78.018478	57.154111	-72.208795	57.925871	-73.166119	58.424077	-74.324713	55.932198	-14.729569	-42.096936	-6.628741	-38.164771	-22.715657	-45.995486	-12.189739	-26.234935	-6.553696	-17.081645	-9.740867	-25.492125	-7.360107	-26.532070	-11.962598	-32.834133	-11.907448	-34.556855	-22.563069	-32.232078	-10.117580	-35.252763	-22.123027	-36.051757	-22.507051	-46.914969	-21.219765	-36.508127	-25.446903	-32.600112	-38.652545	-39.595416	-28.577271	-32.600513	-28.209319	-32.015351	-38.915889	-45.030536	-27.626605	-37.374830	-28.218543	-36.902038	-17.908039	-35.479652	-13.914346	-34.842636	-14.341979	-37.541959	-8.087776	-33.059295	-5.249140	-31.127079	3.469653	-36.909218	-22.516916	-42.061704	0.812134	-21.824458	-3.643922	-24.825354	3.400142	-31.530953	12.225971	-25.901893	33.646940	-25.562584	29.086933	-21.808228	34.918723	-26.822629	29.632528	-20.874425	-3.324419	34.280579	107.229454	26.078995	106.895877	26.639744	108.264742	30.004974	108.308230	29.305926	3.161926	31.101699	2.532747	30.848212	1.382522	29.022500	1.028279	30.206204	1.717944	32.147152	-56.024486	38.019969	-54.098501	39.418451	-53.787429	37.515374	-42.883306	-41.660090	-31.738819	-33.511964	-44.862971	-46.255894	-56.673186	35.431764	-58.072754	35.620179	-38.655707	19.388902	-42.018818	16.310043	-55.781283	36.745618	-52.791441	36.642657	-52.890329	35.254029	-54.847950	37.762164	-53.906567	36.407287	-55.471836	35.255316	1.003197	31.760854	1.232054	31.171443	-55.047113	32.806612	2.463042	30.472300	-6.698041	-23.277562	5.313645	31.337451	4.898307	30.895230	14.269930	28.601033	5.566176	26.653689	-0.509196	26.859393	11.807545	15.737272	20.279144	9.275757	-52.720077	46.318972	-9.143532	19.809020	-8.023651	13.415144	-2.227479	22.336440	-1.202863	39.731207	125.689454	31.802526	123.623912	32.508088	117.156211	33.176282	-37.123288	-54.906026	-43.694688	-45.107174	128.455630	-58.002019	-43.993427	-45.314961	-43.612136	-45.692385	-42.714110	-47.141979	-48.137915	-43.874833	-43.235771	-47.820500	-43.055107	-48.701403	-40.707692	-49.706207	-37.726444	-43.716800	-29.996543	-32.988102	-31.635897	-20.343771	-47.907232	-41.149117	-49.057972	-42.358962	-37.025331	-40.448852	-38.780988	-37.214421	-49.196320	-42.315908	-51.728746	-27.805367	-54.072081	-31.050007	-52.531416	-31.279070	-55.924184	-34.787853	-25.095995	-46.416338	-18.554177	-39.783580	-36.675843	-48.086661	-28.397381	-9.448268	-35.588422	-7.036870	-31.750556	-6.787735	-50.294655	-3.148709	-42.094067	-15.666340	-35.774094	-30.086812	8.969352	-17.360995	12.791630	-19.035628	-23.766145	-35.698835	9.054193	-27.286213	14.294157	-5.216139	-17.636639	-24.501574	-24.635846	17.733481	-27.592024	-30.581245	2.279867	22.968704	-42.772785	-38.924179	-19.057785	-27.373433	-26.152020	-36.074629	-46.533424	-45.427841	57.727544	-23.778935	-46.730798	-44.842577	-54.094146	-42.105197	-45.203055	-45.541631	26.496295	-32.577800	84.126456	35.631481	79.028185	40.669274	29.286472	45.422690	-29.305237	-40.500239	-29.304897	-40.431166	11.243719	43.897285	19.935230	32.462859	-14.036200	-7.739099	8.557910	38.162535	12.278853	42.797551	7.969308	42.878494	6.951825	39.834531	8.901396	39.105408	9.813555	39.526313	89.232192	45.214151	108.981179	24.688302	109.163832	25.532741	108.542067	25.759209	100.202369	32.515806	108.672868	24.909469	-46.806444	55.036178	-26.442127	50.772641	2.678326	45.782293	15.930648	46.302893	29.391303	50.102649	28.167028	44.526409	-57.750182	-32.657167	52.400360	58.538062	3.931317	25.633931	6.675960	28.333080	-13.184932	-13.209546	-16.696279	-24.084090	-16.689247	-20.599335	-42.883544	-46.818603	-45.915808	-46.364264	94.311315	41.864830	85.503992	40.002109	86.982928	53.029807	92.649657	57.296113	97.374688	58.108571	104.550057	60.695695	81.798605	40.773880	80.711390	44.698624	59.544305	44.664317	96.108367	48.631514	-71.799165	46.367413	-74.173873	43.036074	-71.297477	51.448975	-77.151160	50.362740	-77.204014	57.763566	-76.257800	52.044779	-76.159953	50.855669	-81.742385	58.435887	-74.276825	56.387295	96.430326	35.943030	36.734262	69.572319	77.806002	50.404921	93.535824	42.438248	90.506401	41.787942	90.494991	41.250914	97.310766	43.247135	95.392894	41.333276	96.084909	40.987880	93.138490	40.658745	90.779422	40.630597	96.545773	41.810290	119.664959	45.240630	20.274803	62.024273	33.924777	50.339294	35.843983	43.610201	118.947103	45.551024	93.905097	46.603010	-55.942037	-44.987455	-48.469341	-45.438172	89.480007	41.449681	119.688859	42.230679	122.511017	44.336419	123.016076	44.933614	92.577697	41.055327	92.522019	40.734607	89.809634	41.480110	85.839257	40.523498	79.893406	37.785707	-72.241082	58.437998	-79.743034	62.865890	91.101503	41.399355	91.891568	41.638150	90.332344	41.255453	90.375407	40.985314	89.758048	40.416351	90.093397	40.533564	95.572208	53.194543	99.464398	40.730366	99.635065	40.322652	99.436177	39.929495	100.111879	39.946951	97.347436	45.883575	-40.994489	76.258171	120.704740	43.092163	122.348664	44.699864	121.660245	42.992630	122.092092	43.226345	122.171482	43.101488	120.206861	41.788258	120.427498	41.901832	123.800542	44.209603	123.438879	44.245510	118.370452	38.985500	121.318925	39.804003	90.995443	41.359409	92.999094	38.492605	114.355401	43.983732	114.390308	44.872631	-46.447910	-46.791721	-46.630620	-46.932617	-47.705997	-46.923874	114.508777	44.672569	114.645089	45.444898	123.397571	44.655994	98.642325	52.181731	98.083313	54.833741	87.553629	43.228227	90.315380	42.060566	86.389714	46.165755	89.872723	42.884274	120.540175	44.761967	120.884964	44.755653	120.814251	44.671515	121.018320	44.805152	121.084459	44.878291	121.907195	44.219843	121.607164	44.268775	122.519141	44.471801	122.522009	44.469351	120.592298	44.983636	119.984237	44.438959	119.751699	44.424924	118.822422	42.993890	119.954372	44.418156	120.405625	44.795736	120.363543	44.538554	119.597114	43.838166	117.994624	44.837417	118.703079	44.067078	117.591399	43.604737	118.098624	43.445361	116.950552	44.451155	115.670587	44.324141	117.505826	44.978617	114.865731	47.453919	120.426788	44.840806	119.964153	44.918632	120.715697	44.839303	122.367712	43.846233	123.462963	44.216691	121.571967	44.351959	120.651891	44.887553	114.014902	45.361177	107.895500	47.309485	103.268166	46.297512	-57.850985	48.005929	-70.702869	38.084988	-68.935692	38.734778	-63.801139	39.089484	-69.789814	41.963082	-68.914745	42.707646	-68.449492	43.312862	");
//	LoadRatesFromStr("5000000	288.328853	0.009159	2.488875	2.714179	3.828256	2.287889	2.436114	1.973251	2.610829	2.462855	2.366684	1.973528	1.915911	1.513107	1.949874	1.981611	1.845254	2.421675	2.438663	2.637304	2.716137	2.750677	2.692969	2.716249	2.718206	1.473751	1.444184	1.546933	1.345488	1.506435	1.191772	1.394676	1.453141	1.585412	1.311838	1.480633	2.239537	1.863841	1.582355	2.521889	2.460154	2.592014	2.575806	2.816637	2.858764	2.494994	3.068337	2.861372	3.34047	2.175996	1.696778	1.860562	1.654537	1.285238	1.210652	1.015188	0.91011	1.11409	1.286758	1.047825	0.800071	1.307353	1.138519	1.350845	1.455858	1.441877	1.311041	1.401308	1.314259	1.336229	1.41349	1.399126	1.454459	1.489199	1.652721	1.350327	1.808023	1.742999	1.832743	1.622685	1.78563	1.693727	1.79398	1.647075	1.837716	1.79414	1.878611	1.94758	1.834736	1.972279	1.970635	2.264778	1.881545	2.178631	2.602169	2.500089	2.837382	2.825085	2.806964	2.923083	2.894581	2.968518	2.064428	1.891093	2.165527	2.83135	3.769385	3.981791	4.156853	2.683093	2.564882	2.822815	2.318354	1.914291	1.42983	3.518993	3.577619	3.091506	3.271363	3.53834	3.276928	3.606499	3.251272	3.655934	3.390693	3.570037	3.391614	3.273107	3.223141	2.174581	2.105966	1.705942	1.871201	2.141587	2.349143	2.355046	2.485614	2.405754	2.652292	2.709332	2.750151	2.754016	3.07484	3.280935	2.975086	3.013273	3.07358	2.989121	3.230679	3.315549	2.995441	3.405316	3.280307	3.696253	3.830474	3.465136	3.868947	3.861395	3.935801	4.05411	4.006014	4.039399	3.982164	3.791263	3.83816	3.968226	3.875255	3.903599	3.847653	3.488136	3.791121	3.377113	3.229324	3.777927	3.830893	3.82772	3.90478	3.27615	3.885445	3.873099	3.773435	3.904537	3.988921	4.110446	4.488499	4.245838	4.355263	4.255057	3.970921	3.90133	3.888987	4.005717	2.950052	3.208981	3.899755	6.328724	3.824202	4.513047	5.455294	5.224986	3.640851	3.613381	3.642334	3.12964	2.874457	2.384268	2.422745	2.03584	1.881973	1.890797	1.559959	1.547085	1.384456	1.421835	1.3689	1.572464	2.564552	2.034348	1.938359	1.754484	2.049914	1.997939	2.004397	1.755421	2.033655	1.092865	2.430967	2.233602	2.566693	2.408008	2.947896	3.01911	3.504402	4.020505	3.413684	3.695372	3.608074	4.180622	4.092932	1.94134	3.116863	3.425187	3.186558	2.725957	2.717289	2.743878	3.33938	3.269981	3.027702	3.255405	2.910217	2.71203	2.410046	2.513022	2.43887	1.903971	1.520301	1.883268	1.520064	1.347288	1.324283	1.098534	1.32977	1.142124	1.381428	1.866591	1.476635	1.621764	1.448901	1.984072	2.31183	2.245032	2.202679	2.178377	2.29683	2.290049	2.528582	2.405509	2.581221	2.50146	2.521912	2.704181	1.541403	1.581988	1.32717	0.944778	1.319603	1.079948	1.883243	1.290012	1.510336	1.179025	1.329177	1.399776	1.146988	0.727066	1.028973	1.568123	1.299919	0.552584	0.88313	1.122093	1.145754	0.952377	0.841528	1.117876	1.192532	0.994264	1.092281	0.802539	0.966701	0.945302	0.839669	0.82117	0.8118	0.70935	1.416082	0.825442	1.127654	0.870107	1.03057	0.661831	1.19419	1.074857	2.965292	2.994679	3.483155	3.977105	3.527319	3.584723	3.779551	3.875942	3.600567	3.855506	4.534721	4.543124	4.197179	4.188201	4.354411	4.148646	3.67944	3.592273	3.583951	4.143906	3.768068	3.811982	3.534825	3.46987	3.622445	3.519138	3.546955	4.152086	4.233698	4.098076	4.226609	4.10048	4.016219	4.231931	3.951036	4.017774	4.015374	4.239259	4.360552	4.527007	4.537584	4.571744	4.599536	3.439116	3.449352	3.406891	3.350054	3.177855	3.336719	3.501829	3.510148	3.267031	3.14	3.255208	3.396066	4.010964	3.391588	3.446204	3.430027	3.410819	3.560547	3.827218	3.961372	2.940951	2.956639	3.011285	2.909924	3.327	3.394304	3.325797	3.419205	3.3964	3.467836	3.381985	3.08594	3.396044	2.802666	2.878521	2.958037	3.064197	3.073763	2.943718	3.111696	3.165858	3.158301	3.067134	3.040442	3.089068	2.958902	3.195707	3.401997	2.888825	3.070416	2.985022	2.905622	3.903234	3.663285	3.618768	3.543705	3.4646	3.534107	3.69838	3.439259	3.083101	3.433154	3.3138	3.498844	3.440372	3.608005	3.698932	4.018339	4.131731	4.180791	4.148828	4.155728	3.691199	3.756576	3.860385	4.221993	4.489002	4.708629	4.67741	5.184019	4.75228	3.56355	4.530603	5.220056	5.064104	5.098528	4.942626	4.936727	4.952022	5.250718	5.340725	5.268505	5.43427	5.251498	5.516991	5.683141	5.762604	6.281444	5.481649	5.443539	5.517257	4.900764	5.084162	4.996756	4.98568	5.07354	5.075486	4.97012	4.998746	3.631919	3.313503	3.220856	3.364994	3.061171	3.172841	2.971009	3.217174	3.683464	3.499517	3.555442	3.42756	3.349525	3.165912	2.552132	2.534502	2.46022	2.372138	2.391688	2.279393	2.893875	2.913217	2.81115	2.858261	3.182086	3.089216	3.344557	3.422959	3.371829	3.351497	3.351032	3.399583	3.361952	3.071977	2.916587	2.745957	3.567932	4.057832	3.058394	3.355602	3.294619	3.29417	3.331011	3.337424	3.37387	2.625779	3.97401	3.817033	3.944383	3.69299	3.89864	3.901337	3.699108	3.979422	3.114123	4.940323	5.648905	5.663221	5.642607	5.979863	5.225889	5.31036	5.310935	5.329918	6.191	6.107279	5.81668	4.174387	5.627681	5.595859	5.100356	4.228232	3.959525	4.234557	4.250259	4.906137	4.624911	4.962006	4.640088	4.827511	5.02429	4.748401	5.337503	5.697029	5.708959	6.742147	7.025272	7.365861	7.42683	7.414737	7.407938	7.396991	5.719552	6.127467	5.590704	5.991929	5.024087	5.906706	6.325128	5.228112	5.258957	5.183383	5.28181	5.796586	5.237657	4.74738	4.625186	5.339725	4.954325	5.116866	4.959014	4.926243	4.832664	4.777801	4.986087	4.865049	5.192405	5.69623	5.773444	5.776611	5.818697	5.169925	4.47837	3.504204	3.69271	4.805485	5.292095	5.492819	4.748109	4.838423	4.281321	4.136986	4.753604	4.787786	4.919221	4.548926	4.558351	4.484581	4.513301	4.51593	4.558095	4.261139	4.505914	4.203131	4.81465	5.064637	4.399539	5.064236	4.729761	5.073453	5.15251	4.716583	5.175491	5.142026	4.870511	4.998788	5.683374	5.580037	5.084833	5.264724	5.177861	5.327499	5.445266	5.745471	5.896691	5.818511	5.737836	5.808963	5.831892	4.289287	4.302827	4.685351	5.101524	5.368913	5.034078	5.312283	4.810819	5.299002	5.301533	4.834968	4.764399	4.429332	4.386843	4.569099	4.751589	4.568147	4.680156	4.641494	5.033071	4.801645	4.686751	4.917881	4.993577	4.665361	4.664663	4.898144	4.593645	4.44995	4.527467	4.653373	4.534402	4.360135	4.427435	4.349719	4.364284	4.270164	4.287885	4.384796	4.714857	4.717974	4.063407	3.903238	4.083903	4.296171	4.281324	4.29868	4.392782	4.554072	4.085237	4.106291	3.923147	3.941795	4.064781	4.681461	4.749017	4.764411	4.521055	4.889703	4.804172	4.819617	4.752538	4.333462	2.001899	1.57877	1.470016	1.487342	1.737041	1.657038	1.700079	1.574166	1.700285	1.494494	1.389988	1.16325	1.563042	1.093147	1.285999	1.455413	1.451257	1.397538	1.46887	1.615289	1.803784	1.523043	1.609957	2.243954	1.76831	1.900234	1.92181	1.63602	1.498977	1.543104	1.814916	2.081153	1.989369	1.842064	1.868173	1.861609	1.504473	1.537714	2.168964	1.523438	1.093632	1.540229	1.481677	1.567672	1.798434	1.558982	1.964692	1.565388	1.938698	2.358935	2.276719	1.779453	1.810623	2.580789	2.49404	2.506407	2.56216	2.852338	2.789829	2.558808	2.512607	2.892889	2.51855	2.488443	2.559369	2.604994	2.453356	2.543013	2.394414	1.929648	1.896476	1.453427	1.432928	1.320796	1.3122	1.189344	1.115706	1.083463	1.669057	1.626044	1.166139	0.666385	0.706729	0.770933	1.206218	0.750048	1.016491	0.857923	0.920604	0.95844	1.124106	1.328427	0.956711	0.50114	0.81074	0.494205	1.122775	1.398888	0.888205	1.231949	1.340765	1.113406	1.654064	1.346832	1.089813	1.003166	1.049264	0.897672	0.789974	1.152601	0.869533	0.895426	0.91173	0.722766	0.963131	1.054248	1.020831	1.000112	1.022657	0.49443	1.060554	0.995433	1.081958	0.793358	0.957781	0.60583	0.658096	1.03418	1.120607	1.023411	1.089842	1.046433	0.89623	1.288804	0.976767	0.811891	0.722923	0.789269	1.204334	0.80793	1.164508	0.816647	1.177643	1.075499	0.764959	0.959785	1.056175	1.586049	1.084024	1.02754	1.022406	1.056642	1.047421	0.918487	0.9361	0.903725	1.068459	0.640138	0.797024	1.541001	1.675339	1.579323	1.439218	1.46436	1.354629	1.519582	1.692756	1.931315	1.096204	0.907506	1.154111	1.325412	1.180731	0.89295	0.992595	1.092533	0.877584	0.985866	1.142519	1.019727	1.169729	1.007333	0.952033	1.04608	0.965102	1.13795	1.341343	1.256206	0.941407	0.96224	1.350961	1.342724	1.239198	1.528144	1.135076	1.067619	1.242077	1.37191	1.147791	0.964038	1.211392	0.952493	1.29469	0.95408	1.000118	1.40606	1.305875	1.192694	1.468268	1.510399	1.56984	1.594298	1.231853	1.343726	1.30935	1.477012	1.279015	1.284323	1.137793	0.992166	1.378889	1.199859	1.081696	1.306382	1.334317	1.033191	1.477541	1.4033	1.503211	1.179529	1.376544	1.452638	0.980626	0.93014	1.305832	1.325908	1.434419	1.571439	1.2883	1.428538	1.301554	1.390685	1.395323	1.412595	1.378376	1.304097	1.305792	1.273205	1.718516	1.822697	1.591531	1.58188	1.351864	1.461052	1.555077	1.485254	1.450432	1.473261	1.383119	1.588889	0.914861	0.805829	1.062221	0.521035	1.01823	1.193314	1.011273	1.12176	0.56462	0.792729	1.033569	0.885001	0.846469	1.231627	1.022295	1.066897	1.093819	1.231856	0.649191	0.550323	0.646316	1.362901	0.955253	1.021388	1.469559	1.319453	1.619879	1.264651	0.674415	0.635855	0.849385	0.914083	1.083027	1.262708	1.071772	1.228542	1.061983	1.051157	1.137339	1.275581	1.24506	1.266296	1.271727	1.231516	1.220919	0.602261	0.410016	0.61797	0.658997	0.645386	0.742233	0.80434	0.902178	0.841594	1.062508	1.060002	1.093648	1.006119	0.855254	0.99606	1.002667	0.76643	0.95201	1.261533	0.454808	0.953101	0.976167	0.869797	0.677704	1.003734	0.620628	0.575415	0.935557	1.051374	0.968535	1.338	0.877809	0.935406	1.095641	3.044181	3.00726	1.641647	1.539915	2.042289	1.585062	1.830448	1.625608	3.107925	3.12666	3.223167	3.193486	2.557767	2.427467	2.584942	2.353067	2.420533	2.82967	2.100901	2.100189	2.608456	2.611631	2.302992	2.618632	3.24161	2.922991	2.119153	2.052788	2.50678	2.004037	1.899648	1.53904	3.159504	3.423602	3.552795	3.123471	3.58185	2.69558	3.346004	3.347626	3.321325	3.313583	3.362911	3.549689	3.55765	2.038482	3.69057	3.21492	3.263426	3.937601	3.98244	4.017853	4.048299	4.425612	3.896843	3.816529	3.791627	3.762379	3.995593	3.826526	3.335443	3.12408	3.366642	3.407069	3.53001	3.000332	3.2943	3.041053	2.966684	2.96513	2.940803	3.029888	3.096614	3.010787	2.951308	2.924965	3.057771	3.051717	3.341321	3.498719	3.521909	2.832775	2.878584	2.884819	3.021115	3.024264	2.822809	2.586378	2.877696	2.560375	2.659782	2.631145	2.739778	2.718856	2.610422	2.753738	2.935463	2.946579	2.940594	2.641076	2.3463	2.344883	2.692081	2.563931	2.565752	2.522621	2.557485	2.480369	3.600757	3.991126	4.408179	4.126422	4.418662	4.547348	3.786165	3.695181	3.700932	3.743701	3.79342	3.760141	3.90296	3.962509	3.932102	4.036678	4.029628	3.997979	3.734698	3.997257	4.149939	3.998958	3.837906	3.980096	3.994007	4.147309	4.140488	3.922917	3.88538	3.739087	3.678093	3.843902	3.942858	3.8569	3.69058	3.726701	3.782284	3.741762	3.691323	3.73779	3.760424	3.643098	3.654814	3.598694	3.743131	3.478845	3.539205	3.35236	3.685176	3.83251	3.830079	4.098456	4.032622	4.014288	4.013793	4.129258	4.052956	4.03432	3.825639	3.851393	3.907384	4.009233	4.051068	4.10391	4.009329	3.960562	3.67539	3.785662	3.852302	3.080547	2.478201	2.450115	2.130141	1.765028	1.694884	1.852551	2.315806	1.818285	2.158008	2.176325	2.160224	3.057682	3.47028	3.409992	3.031156	3.010566	3.145917	3.00383	2.972189	2.991963	3.021807	2.653776	2.987536	2.975598	2.900719	2.918501	3.228798	3.161463	3.227897	3.122151	3.23143	3.352306	3.280025	3.240888	3.016151	3.239878	3.378292	3.27722	3.191641	3.393824	3.336473	3.399181	3.342429	3.389784	3.431328	3.395857	3.45507	3.386907	3.42297	3.391699	3.39347	3.392601	3.575548	3.533943	3.567446	2.641432	2.44262	2.114553	1.777585	1.678287	1.838757	1.64987	1.69321	1.810777	1.806837	1.552074	2.916905	2.749161	2.698883	2.398962	2.411459	2.544742	2.745947	2.83678	2.775417	2.687545	2.524639	2.52196	2.76374	3.03309	2.816177	3.143458	2.858815	3.103029	2.384924	2.137008	2.009938	2.106576	2.096122	2.023238	2.007612	2.016415	2.085326	2.422149	2.259172	2.071547	2.59429	2.106004	2.096224	2.124499	2.585988	2.594737	2.577883	2.584221	2.570674	2.553359	2.544423	2.535455	2.551197	2.547351	2.517564	2.548633	2.568474	2.564636	2.618986	2.645132	2.55106	2.571009	2.526495	2.553031	2.619064	2.672814	2.748129	2.60241	2.525513	2.390789	2.610879	2.648698	2.665768	2.599887	2.626445	2.616826	1.786561	1.752758	1.650547	2.054615	1.970385	2.164949	1.883635	1.451598	2.094596	2.662185	3.04403	2.972689	2.490982	2.034446	2.126305	2.494663	2.175247	2.898361	2.610798	2.283942	2.100763	2.172736	2.575074	2.072492	1.698915	1.679644	1.758548	1.540766	1.777746	1.964229	1.811303	1.88227	1.751036	1.985624	2.188818	1.954941	1.759871	1.720797	1.801659	1.663939	1.939096	1.893995	1.801717	1.832966	1.848196	1.804633	2.538405	2.037189	2.821399	3.378882	3.402447	3.390795	3.616603	3.618228	2.849498	2.898342	2.938736	2.967116	2.622414	2.463651	2.480098	2.530987	2.487429	2.431122	2.512463	2.436239	2.459522	2.713746	2.694735	2.714152	2.636504	2.45573	2.265047	2.389633	2.469485	2.43473	2.403421	2.400821	2.359274	2.441696	2.481539	2.589617	2.912863	2.767105	2.217683	2.55904	2.407748	2.429751	2.705071	1.816898	2.767092	2.463932	2.726691	2.271639	2.527388	2.182041	2.528708	3.168074	3.258669	3.122671	3.196541	3.498005	3.323163	2.675779	2.584372	2.113143	2.738973	2.371364	2.374592	2.730027	2.445243	2.80879	2.533891	2.641076	3.069448	3.945128	3.48608	3.508487	3.927548	3.382665	2.805639	2.705021	2.181479	2.138632	2.721134	2.292844	3.279215	3.210883	3.139264	3.132057	3.554256	3.188822	2.699931	2.295414	2.800597	2.81466	4.433173	1.397184	1.719093	1.913127	2.138264	1.993216	1.903645	2.14794	2.287846	2.247758	2.379382	2.524266	2.308939	2.480742	2.499216	2.799759	2.381136	2.356775	2.11956	1.776742	1.931844	1.982697	1.953078	1.929628	1.771919	2.015756	1.846199	1.124471	1.071664	0.882485	1.45261	1.316747	1.43067	0.943144	1.10183	1.107278	1.082378	1.006319	1.312942	1.174358	1.2695	1.030932	1.909992	1.914772	1.959812	1.855748	1.895497	2.023818	1.907408	1.89978	1.585737	1.610695	1.951128	1.738108	1.84509	1.842109	1.86458	2.880286	2.737796	2.036271	1.762114	2.024987	2.483432	2.348363	1.662598	1.92248	1.633804	1.64897	1.274123	2.170293	2.175575	1.426258	1.318585	1.805702	1.823975	2.08184	1.699847	1.702106	1.530561	1.598902	1.723258	2.303382	1.761552	3.063341	1.64465	2.085041	1.086545	1.868912	1.268485	3.206292	1.976815	2.758743	1.540398	1.143778	1.702466	1.429001	1.779498	1.669594	1.953022	1.992236	2.246294	1.955615	2.596272	1.754435	2.106335	1.959561	1.491381	1.900123	2.086418	2.556344	2.418479	2.102166	1.978019	2.663501	1.932341	1.531493	1.727647	1.594064	2.02783	2.024775	1.489725	1.601669	1.681863	1.962146	2.316045	1.862737	1.304942	1.829519	2.220902	1.897085	1.967048	1.898283	1.663221	1.625396	1.861048	0.979001	1.247642	0.90893	2.023488	1.941846	1.872489	1.92293	1.794402	2.195959	1.995927	2.09712	1.952561	3.131706	1.863816	1.240044	2.342719	2.302001	2.349208	2.089456	1.696148	2.236627	1.767458	1.894641	1.738399	2.454647	2.515546	2.684496	2.546375	2.129842	2.364465	1.640314	1.409969	1.322446	1.102633	0.973488	1.472323	1.113738	1.375691	1.481579	1.210797	1.018274	1.170368	1.00272	1.652912	1.607633	1.391675	1.668125	1.815265	1.889737	1.632391	1.59918	1.60146	2.081546	2.033069	2.018792	2.17997	2.178135	1.741803	1.903483	1.998012	1.564835	1.815908	1.459024	1.637448	1.758174	1.74428	1.798407	1.772692	1.754942	1.570446	1.424266	1.360086	1.517918	1.528473	1.594205	1.360996	1.406422	1.746113	1.71965	1.637867	1.44489	1.672059	1.720909	1.794521	1.891561	1.992588	1.940881	1.770101	1.556848	1.696256	1.772899	1.169367	2.002531	1.924383	2.198287	1.873368	1.385447	2.073173	1.800007	1.466019	1.822308	1.823853	1.740455	1.553563	1.898027	1.726659	1.928933	1.441906	1.609996	1.576838	1.428858	1.565301	1.332564	1.430038	1.240117	1.468241	1.27026	1.369215	1.495837	2.223442	1.470386	1.648409	1.628353	1.508848	1.418949	1.164689	1.532683	1.758491	1.612448	1.610978	1.693305	1.900156	1.501576	1.604071	1.797371	1.599293	1.591632	1.645477	1.300615	1.307882	1.707649	1.355027	1.417227	1.52338	1.411784	1.555471	1.637536	1.897719	1.749276	1.377248	1.320646	1.495031	1.817703	2.106496	1.784606	2.524566	2.802686	2.065375	1.946157	1.973803	1.873459	1.753765	2.319649	2.681222	2.642835	2.685164	2.581282	3.187069	3.193465	0.83629	0.913617	1.026765	1.612573	1.580452	1.923644	2.234628	2.127584" , CRates, Opt, Trees);

	CRates->Lh	=	Likelihood(CRates, Trees, Opt);
	CalcPriors(CRates, Opt);

	if(Opt->UseSchedule == TRUE)
		ShedFile = SetScheduleFile(Opt, Shed);

	if(Opt->SaveModels == TRUE)
		SaveModelF = InitSaveModelFile(Opt->SaveModelsFN, Opt, Trees, CRates);

	StoneF = NULL;
	if(Opt->Stones != NULL)
		StoneF = CreatStoneOuput(Opt);

	GBurntIn = BurntIn = FALSE;
	if(Opt->BurnIn == 0)
		BurntIn = TRUE;

//	MCMCTest(Opt, Trees, CRates, Shed);
	StartT = GetSeconds();

	for(Itters=0;;Itters++)
	{

		SetCustomSchedule(Opt, ShedFile, Itters, Shed);

 		CopyRates(NRates, CRates, Opt);

		MutateRates(Opt, NRates, Shed, Itters);

		if(Opt->NodeData == TRUE)
			SetTreeAsData(Opt, Trees, NRates->TreeNo);

		NRates->Lh = Likelihood(NRates, Trees, Opt);

		if(NRates->Lh == ERRLH)
			Itters--;
		else
		{
			CalcPriors(NRates, Opt);

			if(MCMCAccept(Itters, Opt, Trees, Shed, CRates, NRates) == TRUE)
			{
				Swap((void**)&NRates, (void**)&CRates);
				UpDateShedAcc(TRUE, Shed);
			}
			else
				UpDateShedAcc(FALSE, Shed);

			if(Itters % Opt->Sample == 0)
			{
				// Needed to make sure the tree has correct values set
				// Also a sandity check.
				if(CRates->Lh != Likelihood(CRates, Trees, Opt))
				{
					printf("Likelihood Error: %d::%s\n", __LINE__, __FILE__);
					exit(1);
				}

				PrintStdOut(Opt, CRates, Itters, GetSeconds() - StartT, BurntIn);
			}


			if( (Itters % Opt->Sample) == 0 &&
				BurntIn == TRUE
				&& StonesStarted(Opt->Stones, Itters) == FALSE)
			{
				UpDateHMean(Opt, CRates);

				PrintMCMCSample(Itters, Shed, Opt, CRates, Opt->LogFile);
				fflush(Opt->LogFile);

				if(Opt->UseSchedule == TRUE)
					PrintShed(Opt, Shed, ShedFile);

				if(UseNonParametricMethods(Opt) == TRUE)
					PrintVarRatesOutput(Opt, Trees, CRates, Itters);

				if(Opt->ModelType == MT_FATTAIL)
					OutputFatTail(Itters, Opt, Trees, CRates);

				if(Opt->RJDummy == TRUE)
					PrintRJDummy(Itters, Opt, Trees, CRates);

				#ifdef JNIRUN
					fgets(Opt->LogFileBuffer, LOGFILEBUFFERSIZE, Opt->LogFileRead);
					AddResults(Env, Obj, Opt);
					SetProgress(Env, Obj, Itters);
				#endif

				if(Opt->SaveModels == TRUE)
					SaveModelFile(SaveModelF, Opt, Trees, CRates);

				if(Opt->SaveTrees == TRUE)
					OutputTree(Opt, Trees, CRates, Itters, Opt->OutTrees);
			}

			if(Itters % MCMC_SCHEDULE_UPDATE == 0)
			{
				// The schedule should be updated even when stones are running
				UpDateSchedule(Opt, Shed, CRates->RS);
				BlankSchedule(Shed);
			}

			if(ExitMCMC(Opt, Itters) == TRUE)
			{
				if( (Opt->UseEqualTrees == FALSE) ||
					(CRates->TreeNo == Trees->NoTrees - 1))
				{
					EndT = GetSeconds();
					printf("Sec:\t%f\n", EndT - StartT);

					FreeRates(CRates, Trees);
					FreeRates(NRates, Trees);

					FreeeSchedule(Shed);

					if(StoneF != NULL)
						fclose(StoneF);

					if(Opt->UseSchedule == TRUE)
						fclose(ShedFile);

					if(UseNonParametricMethods(Opt) == TRUE)
						FinishVarRatesFiles(Opt);

					if(SaveModelF != NULL)
						fclose(SaveModelF);
					return;
				}

				if(GBurntIn == TRUE)
				{
					CRates->TreeNo++;
					CRates->Lh = Likelihood(CRates, Trees, Opt);
					Itters = 0;
					BurntIn = FALSE;
					BlankSchedule(Shed);
					Shed->GNoAcc = Shed->GNoTried = 0;
				}
			}

			#ifdef JNIRUN
				if(Itters%100==0)
				{
					CheckStop(Env, Obj, Trees);
					if(Trees->JStop == TRUE)
					{
						FreePriors(CRates);
						FreePriors(NRates);

						FreeRates(CRates);
						FreeRates(NRates);

						free(Shed);

						return;
					}
				}
			#endif

			if(Opt->UseEqualTrees == TRUE)
			{
				if((Itters == Opt->ETreeBI) && (GBurntIn == TRUE))
					BurntIn = TRUE;
			}

			if(Itters == Opt->BurnIn)
			{
				if((Opt->UseEqualTrees == TRUE) && (GBurntIn == FALSE))
				{
					GBurntIn = TRUE;
					Itters = 1;
					BurntIn = FALSE;
				}
				else
					BurntIn = TRUE;
			}

			if(Opt->Stones != NULL)
				StoneItter(Opt->Stones, Itters, CRates->Lh, StoneF);
		}
	}
}

